#!/usr/bin/env python3

"""Lookup DMR users on radioid.net

This script fetches repeater information from radioid.net API and
prints them to stdout.

Author:        Dominic Reich (OE7DRT) <quick.hat4396@qtztsjosmprqmgtunjyf.com>
Created:       2022-08-27 12:08:11+0200
Last modified: 2022-08-27 12:31:42+0200

Usage:
------
    With arguments
    ./rpt.py [callsign]|[dmrid]....

    Without arguments - but enter them when the script asks you for them
    ./rpt.py


Examples:
---------
no examples yet...
"""

import re  # regex
import sys
import requests


def printFormatted(data: dict) -> None:
    """Formats and prints to stdout.

    Expects data coming in as dict, returns nothing but
    prints the text to stdout."""
    """    id      call    city     state    country  color  freq   offset trustee """
    print("{:>7.8} {:<8.8} {:16.16} {:23.23} {:13.13} {:2.2} {:9.9} {:6.6} {:8.8}".format(
        str(data['id']         or ''),
        str(data['callsign']   or ''),
        str(data['city']       or ''),
        str(data['state']      or ''),
        str(data['country']    or ''),
        str(data['color_code'] or ''),
        str(data['frequency']  or ''),
        str(data['offset']     or ''),
        str(data['trustee']    or '')))


def printHeaderLine():
    """Prints the first line of table """
    print("  DMRID CALLSIGN CITY             STATE                   COUNTRY       CC FREQUENCY OFFSET TRUSTEE")


def getFromCallsign(url: str, call: str) -> None:
    """Maps the given callsign to a DMRID

    Uses the given base-url and callsign and fetches the
    connected DMRIDs. Loops over the result and executes
    printFormatted to print the results to stdout."""
    r = requests.get(url + "/api/dmr/repeater/?callsign={}".format(call))
    for ids in range(0, r.json()['count']):
        printFormatted(r.json()['results'][ids])


def main():
    """main function

    Runs if script is run by itself."""
    if len(sys.argv) <= 1:
        # Check if arguments were given, if not, let the user
        # input some here
        arguments = []

        userinput = input("No arguments found, enter them now: ")
        words = userinput.split()
        for word in words:
            arguments.append(word)
    else:
        # If arguments were given, take them and move on
        arguments = sys.argv[1:]

    # Using these regex pattern to match agains valid DMRIDs
    # and callsigns. The callsign pattern was taken from
    # <https://gist.github.com/JoshuaCarroll/f6b2c64992dfe23feed49a117f5d1a43>
    # and slighty modifed to also allow the percent sign (%) used by
    # used by radioid.net as wildmask.
    dmrid_userid_patt = re.compile("^[0-9]{7}$")
    dmrid_rep_patt = re.compile("^[0-9]{6}$")
    # dmrid_other_patt = re.compile("^[0-9]{0,5}$")
    call_patt = re.compile("^[a-zA-Z0-9%]{1,3}[0123456789%][a-zA-Z0-9%]{0,2}[a-zA-Z%]$")

    baseurl = 'https://www.radioid.net'

    printHeaderLine()

    for arg in arguments:
        if dmrid_rep_patt.match(arg):
            # A valid DMRID was found, so we have to lookup the
            # callsign and we may get more DMRIDs for this RPT
            # Valid means 6 chars long, all numbers. We do not know
            # yet if the id really exists.
            r = requests.get(baseurl + '/api/dmr/repeater/?id={}'.format(arg))
            if r.status_code == 200:
                # Only fetch more DMRIDs if the first one exists,
                # otherwise we would try to run code on an non
                # existing variable
                getFromCallsign(baseurl, r.json()['results'][0]['callsign'])

        elif dmrid_userid_patt.match(arg):
            # A valid user ID was found. Valid means, technically
            # correct -> 7 characters long, all numbers
            print("{} looks like a user. Skipping for now.".format(arg))

        # elif dmrid_other_patt.match(arg):
        #     # Print a warning for numbers less than 6 characters
        #     print("{} is not a valid dmr id!".format(arg))

        elif call_patt.match(arg):
            getFromCallsign(baseurl, arg)

        else:
            print('{} is an invalid value'.format(arg))


if __name__ == "__main__":
    main()
