#!/usr/bin/env python3

"""Fetches the DMRID database from radioid.net

This script fetches the database of DMRIDs directly from
radioid.net and creates a CSV file to import it into
the GD77 running OpenGD77.

Author: Dominic Reich “OE7DRT”
        <quick.hat4396@qtztsjosmprqmgtunjyf.com>


Usage:
------
    ./dmrids.py

    No arguments are needed. Specify the folders in the
    source below.
"""

import os
from datetime import datetime, timedelta
import requests
import pandas as pd


def main():
    """main function

    Runs if script is run by itself.
    """

    # User configuration --------------------------------
    database_url = 'https://www.radioid.net/static/user.csv'
    local_folder = '/home/dominic/.hamradio/'
    output_filename = 'DMR-IDS-GD77.csv'
    tmpfile = '/tmp/dmrids.download'
    regions = ['232', '262', '263', '264', '228', '222']
    # regions = ['232']
    fav_filename = 'Favorites.txt'
    # blacklist_filename = 'BlacklistedCalls.txt'

    # End of User configuration -------------------------

    if os.path.isfile(local_folder + output_filename) and os.access(local_folder + output_filename, os.R_OK):
        print('Local file found. Will overwrite the file if it\'s old enough.')
        one_day_ago = datetime.now() - timedelta(days=1)
        filetime = datetime.fromtimestamp(
            os.path.getctime(local_folder + output_filename))
        if not filetime < one_day_ago:
            print('Nope, the file is quite actual ({})\n'.format(filetime))
            a = input(
                'Do you really want to fetch the database? (y/n): ')
            if a.lower() != 'y':
                exit(0)

    # read favorites
    favorites = []
    if os.path.isfile(local_folder + fav_filename) and os.access(local_folder + fav_filename, os.R_OK):
        with open(local_folder + fav_filename, 'r') as file:
            for line in file:
                line = line.strip()
                favorites.append(line)
    else:
        # This is because the program will create weird resuls otherwise.
        # Needs some error handling but is not finished yet. So there we use this
        # now until fixed.
        print('Could not open favorites file.')
        exit(1)

    print('Downloading database (this may take some time)')
    r = requests.get(database_url)
    if r.status_code == 200:
        with open(tmpfile, "wb") as file:
            file.write(r.content)
    else:
        print('Download failed. Exiting')
        exit(1)

    # Damn this took so long to figure out that I have to escape them with \\
    # and not only with \ ... *grml*
    searchstr = '\\b(?:' + '|'.join(regions) + ')\\d+\\b'
    favsearchstr = '|'.join(favorites)

    # Open full csv file and create two DataFrames with filtered users and favorites
    csv_full = pd.read_csv(tmpfile, dtype=str, usecols=['RADIO_ID', 'CALLSIGN', 'FIRST_NAME'])
    csv_filt = csv_full[csv_full['RADIO_ID'].str.contains(searchstr)]
    csv_favs = csv_full[csv_full['CALLSIGN'].str.contains(favsearchstr, na=False)]

    # Merge regional and whitelist, sort and remove dupes
    csv_fin = pd.concat([csv_filt, csv_favs], ignore_index=True)
    csv_fin.sort_values('RADIO_ID', inplace=True)
    csv_fin.drop_duplicates(subset=['RADIO_ID'], inplace=True)

    # Write the final csv file
    csv_fin.to_csv(local_folder + output_filename, index=False)
    # print(csv_fin[csv_fin['CALLSIGN'].isin(['DD7MH'])])


if __name__ == "__main__":
    main()
